/*
 * tvflash - update firmware flash memory on Mellanox HCAs
 * Copyright (c) 2004 Topspin Communications.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * $Id: firmware.h 9509 2006-09-18 01:22:23Z roland $
 */

#ifndef _FIRMWARE_H_
#define _FIRMWARE_H_

#include <stdint.h>
#include <endian.h>

enum operation {
	OP_NONE,
	OP_IDENTIFY,
	OP_PRINT_GUIDS,
	OP_DOWNLOAD,
	OP_UPLOAD, 
	OP_MODIFY_OPTIONS,
};

enum identify_mode {
	IDENTIFY_EXTENDED,
	IDENTIFY_PRIMARY_FIRMWARE_LABEL,
	IDENTIFY_HARDWARE_LABEL,
};

#define TV_FLASH_DEFAULT_SECTOR_SIZE	0x010000	/* 64K */

struct topspin_vsd {
	uint32_t flags;
/*
 * Note: The first shipped version had an endian problem with the
 * flags. So it has been changed to make sure we maintain the
 * endianness and compatibility to the first versions
 */
#define VSD_FLAG_AUTOUPGRADE	0x01000000
#define VSD_FLAG_ALIGN_FIXED	0x00000001	/* Alignment fixed */
#define VSD_FLAG_NEW_BUILD_NUM	0x00000002	/* Use new build num field */
#define VSD_FLAG_BOOT_OPTIONS	0x00000004	/* Boot options supported */
#define VSD_FLAG_BOOT_GID_BIG	0x00000008	/* GID is in big endian */

#define VSD_FLAG_BOOT_OPTIONS_MASK	0x00FF0000
#define VSD_FLAG_BOOT_ENABLE_PORT_1	0x00010000
#define VSD_FLAG_BOOT_ENABLE_PORT_2	0x00020000
#define VSD_FLAG_BOOT_ENABLE_SCAN	0x00040000
#define VSD_FLAG_BOOT_WAIT_ON_ERROR	0x00080000
#define VSD_FLAG_BOOT_TRY_FOREVER	0x00100000

#define VSD_FLAG_BOOT_TYPE		0x00C00000
#define VSD_FLAG_BOOT_TYPE_WELL_KNOWN	0x00000000
#define VSD_FLAG_BOOT_TYPE_SAVED	0x00400000
#define VSD_FLAG_BOOT_TYPE_PXE		0x00800000
#define VSD_FLAG_BOOT_TYPE_DISABLE	0x00C00000

	uint64_t createtime;
	uint64_t flashtime;

	uint8_t  major_ver;	/* 1 */
	uint8_t  minor_ver;	/* 18 */
	uint16_t micro_ver;	/* 0 */
	int8_t   revision_ver;	/* 7 (highbit set for rc) */
	uint8_t  old_build_num;	/* Old build number */
	/* FIXME: This is in little endian when it should be big endian */
	uint16_t build_num;	/* For Topspin use (eg 345) */

	char     hw_label[64];	/* Text label unique to this hardware */

	char     build_rev[16];	/* For Topspin use (eg 2.0.0) */

	uint8_t  unused[0xc2 - 0x70];	/* Program as 0 */

#define BOOT_PXE_SECS_DEFAULT	20
	uint8_t  boot_pxe_secs;		/* Number of secs to wait for DHCP (0 = inf) */
	uint8_t  boot_ioc_num;		/* Saved IOC num */
	uint8_t  boot_service_name[8];	/* Saved service name */
	uint8_t  boot_dgid[16];		/* Saved destination GID */
	uint8_t  boot_port;		/* Port number to use (0 == any) */
#define BOOT_VERSION	2
	uint8_t  boot_version;		/* Version of boot firmware options */

	uint16_t signature2;
} __attribute__((packed));

union vsd {
	unsigned char raw[0x100 - 0x20];

	struct {
#define VSD_SIGNATURE_TOPSPIN 0x5ad
		uint16_t signature;
		uint16_t checksum;    /* Calculated over VSD with field bset to 0 */
		union {
			struct topspin_vsd topspin;
		} vendor;
	} data;
};

struct failsafe {
	unsigned char valid;

	struct image {
		unsigned char valid;

		uint32_t addr;
		uint32_t size;

		union vsd vsd;
	} images[2];
};

enum method {
	METHOD_MMAP,
	METHOD_PCI_CFG,
};

struct pciid;

enum flash_command_set {
	CS_UNKNOWN = 0,
	CS_INTEL = 1,
	CS_AMD = 2,
	CS_SPI = 3,
};

enum spi_sp_type {
	SP_UNKNOWN   = 0,
	SP_ST_M25P80 = 1,	/* ST-25P80 1MByte  serial prom */
	SP_ST_M25P16 = 2,	/* ST-25P16 2MBytes serial prom */
};

#define GPIO_DAT	0xF0080
#define GPIO_DIR	0xF0088
#define GPIO_POL	0xF0090
#define GPIO_MOD	0xF0098
#define GPIO_LOCK	0xF00EC

#define GPIO_DATACLEAR	0xF00F4
#define GPIO_DATASET	0xF00DC

struct tvdevice {
	struct tvdevice *next;

	struct pci_dev *pdev;
	struct pciid *pciid;
	struct board *board;
	unsigned char revision;

	unsigned char num;
	unsigned char flags;

	enum method method;
	uint8_t *bar0;

	unsigned char flash_bank;
	struct failsafe failsafe;

	unsigned flash_size;

	enum flash_command_set flash_command_set;
	unsigned char flash_bank_shift;
	unsigned int flash_bank_mask;
	unsigned int flash_sector_sz;
	unsigned char flash_num_spis;
	unsigned char flash_spi_sp_size;
	enum spi_sp_type flash_spi_sp_type;

	unsigned int gpio_data[2], gpio_direction[2];
	unsigned int gpio_polarity[2], gpio_output_mode[2];

	unsigned char vpd_present;

	/* This is just so we can force alignment for this data */
	union {
		unsigned char vpd_char[256];
		unsigned int vpd_int[256/4];
	} vpd;
};

enum hca_chip {
	CHIP_TAVOR,
	CHIP_ARBEL,
	CHIP_SINAI,
};

/* Board identification */
struct board {
	char *name;
	enum hca_chip chip;
	char *fwlabel;
	int num_ports;
};

/* Tavor */
struct board jaguar = { "Jaguar", CHIP_TAVOR, "HCA.Jaguar", 2 };
struct board cougar = { "Cougar", CHIP_TAVOR, "HCA.Cougar", 2 };
struct board cougarcub = { "Cougar Cub", CHIP_TAVOR, "HCA.CougarCub", 2 };
/* Arbel */
struct board lioncub = { "Lion Cub", CHIP_ARBEL, "HCA.LionCub", 2 };
struct board lioncub_revc = { "Lion Cub", CHIP_ARBEL, "HCA.LionCub.RevC", 2 };
struct board lioncub_ddr = { "Lion Cub DDR", CHIP_ARBEL, "HCA.LionCub-DDR", 2 };
struct board glacier = { "DLGL", CHIP_ARBEL, "HCA.DLGL", 2 };
struct board bc2 = { "BC2 HSDC", CHIP_ARBEL, "HCA.HSDC", 2 };
struct board lionmini = { "Lion Mini", CHIP_ARBEL, "HCA.LionMini", 2 };
struct board lionmini_ddr = { "Lion Mini DDR", CHIP_ARBEL, "HCA.LionMini-DDR", 2 };
struct board genarbel = { "Generic Arbel", CHIP_ARBEL, NULL, 2 };
/* Sinai */
struct board tiger = { "Tiger", CHIP_SINAI, "HCA.Tiger", 1 };
struct board cheetah = { "Cheetah", CHIP_SINAI, "HCA.Cheetah", 1 };
struct board cheetah_ddr = { "Cheetah DDR", CHIP_SINAI, "HCA.Cheetah-DDR", 1 };
struct board gensinai = { "Generic Sinai", CHIP_SINAI, NULL, 1 };

/* Map VPD string to board */
struct board_vpd {
	char *str;

	struct board *board;
};

struct board_vpd board_vpds[] = {
	{ "Cougar cub", &cougarcub },
	{ "Lion cub", &lioncub },
	{ "Lion cub DDR", &lioncub_ddr },
	{ "Lion mini", &lionmini },
	{ "Lion mini DDR", &lionmini_ddr },
	{ "DLGL", &glacier },
	{ "Tiger", &tiger },
	{ "Cheetah", &cheetah },
	{ "Cheetah DDR", &cheetah_ddr },
	{ "BC2 HSDC", &bc2 },
};

/* Map PCI id to chip and function to identify board */
#define PCI_VENDOR_MELLANOX			0x15b3
#define PCI_VENDOR_TOPSPIN			0x1867

#define PCI_DEVICE_MELLANOX_MT23108		0x5a44
#define PCI_DEVICE_MELLANOX_MT23108_PCICONF	0x5a45
#define PCI_DEVICE_MELLANOX_MT23108_PCIBRIDGE	0x5a46
#define PCI_DEVICE_MELLANOX_MT25208		0x6282
#define PCI_DEVICE_MELLANOX_MT25208_COMPAT	0x6278
#define PCI_DEVICE_MELLANOX_MT25208_PCICONF	0x6279
#define PCI_DEVICE_MELLANOX_MT25204		0x6274
#define PCI_DEVICE_MELLANOX_MT25204_PCICONF	0x6275

struct pciid {
	uint16_t vendor;
	uint16_t device;
	unsigned int flags;

	struct board *(*identify)(struct tvdevice *);
};

static struct board *tavor_identify(struct tvdevice *);
static struct board *arbel_identify(struct tvdevice *);
static struct board *sinai_identify(struct tvdevice *);

#define FLAG_RECOVERY		1
#define FLAG_TAVOR_COMPAT	2

struct pciid pciids[] = {
	/* Tavor, only one has VPD */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT23108,
		0, tavor_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT23108,
		0, tavor_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT23108_PCICONF,
		FLAG_RECOVERY, tavor_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT23108_PCICONF,
		FLAG_RECOVERY, tavor_identify },

	/* Arbel, all have VPDs */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25208,
		0, arbel_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25208,
		0, arbel_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25208_COMPAT,
		FLAG_TAVOR_COMPAT, arbel_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25208_COMPAT,
		FLAG_TAVOR_COMPAT, arbel_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25208_PCICONF,
		FLAG_RECOVERY, arbel_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25208_PCICONF,
		FLAG_RECOVERY, arbel_identify },

	/* Sinai, all have VPDs */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25204,
		0, sinai_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25204,
		0, sinai_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25204_PCICONF,
		FLAG_RECOVERY, sinai_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25204_PCICONF,
		FLAG_RECOVERY, sinai_identify },
	/* Some old Tiger and Cheetah boards use PCI device id 0x538x (24204) */
	{ PCI_VENDOR_MELLANOX, 0x5e8c,
		0, sinai_identify },
	{ PCI_VENDOR_MELLANOX, 0x5e8d,
		FLAG_RECOVERY, sinai_identify },
};

#define GUID_LEN		8

#define TV_FLASH_GUID_OFF	0x24

#define MAX_STR			0xB0

#endif
